
#include "ewrte.h"
#include "ewgfx.h"

/* Include VC++ header files needed in this example. */
#include <stdio.h>
#include <wchar.h>
#include <io.h>
#include <windows.h>


EW_MODULE
( 
  INTRINSICS_IFC_VERSION,
  L"IntrinsicsModule", 
  L"Intrinsics module to access the file system"
)


/* Following variables will store the collected file names. */
static int       NoOfFiles = 0;
static wchar_t** Files     = 0;


/* The function frees the information collected by the preceding OpenFolder()
   invocation. */
static void CloseFolder( void )
{
  int fileNo = 0;

  /* Release the memory reserved for the collected file names. */
  for ( ; fileNo < NoOfFiles; fileNo++ )
    if ( Files[ fileNo ])
      EwFree( Files[ fileNo ]);

  /* Release the memory reserved for the array with file names. */
  EwFree( Files );

  /* No collected data anymore. */
  Files     = 0;
  NoOfFiles = 0;
}


/* The function evaluates the given folder and collects all files matching 
   aPattern in the global array 'Files'. The folder is searched relative to
   the directory where the Embedded Wizard project is found. */
static void OpenFolder( XString aFolder, XString aPattern )
{
  wchar_t             path[ MAX_PATH ];
  intptr_t            handle;
  struct _wfinddata_t fileInfo;
  int                 fileNo = 0;

  /* First ensure that names of evtl. previously collected files are freed. */
  CloseFolder();

  /* The folder is searched relative to the current working directory. This
     corresponds to the directory of the opened Embedded Wizard project.
     Limit to files matching the pattern. */
  wcscpy_s( path, MAX_PATH, aFolder );
  wcscat_s( path, MAX_PATH, L"\\" );
  wcscat_s( path, MAX_PATH, aPattern );

  /* Phase 1: count the files matching the pattern. */
  if (( handle = _wfindfirst( path, &fileInfo )) != -1 )
  {
    do
    {
      /* Is this a file? */
      if ( !( fileInfo.attrib & _A_SUBDIR ))
        NoOfFiles++;
    }
    while (( _wfindnext( handle, &fileInfo ) == 0 ));

    /* Terminate the search operation and release unused memory. */
    _findclose( handle );
  }

  /* Phase 2: Allocate memory for an array containing the file names. */
  Files = EwAlloc( NoOfFiles * sizeof( XString ));
  memset( Files, 0, NoOfFiles * sizeof( XString ));

  /* Phase 3: Collect the file names matching the pattern. */
  if (( handle = _wfindfirst( path, &fileInfo )) != -1 )
  {
    do
    {
      /* Is this a file? Collect it in the array 'Files' */
      if ( !( fileInfo.attrib & _A_SUBDIR ) && ( fileNo < NoOfFiles ))
      {
        int len = wcslen( fileInfo.name );

        Files[ fileNo ] = EwAlloc(( len + 1 ) * sizeof( wchar_t ));
        wcscpy_s( Files[ fileNo++ ], len + 1, fileInfo.name );
      }
    }
    while (( _wfindnext( handle, &fileInfo ) == 0 ));

    /* Terminate the search operation and release unused memory. */
    _findclose( handle );
  }
}


/* The following function just returns how many files have been collected by the
   preceding OpenFolder() invocation. */
static XInt32 GetNoOfFiles( void )
{
  return NoOfFiles;
}


/* The following function returns the collected file with the given number. If
   the file is not available, the function returns an empty string. */
static XString GetFileName( XInt32 aFileNo )
{
  /* The requested file is not existing. Return an empty string. */
  if (( aFileNo < 0 ) || ( aFileNo >= NoOfFiles ))
    return 0;

  /* Return an Embedded Wizard string containing a copy of the collected file
     name. */
  return EwNewString( Files[ aFileNo ]);
}


/* The following function opens the given text file, reads its content and 
   returns it as a string. The parameter aFilePath specifies the location of
   the file relative to the directory where the Embedded Wizard project is 
   found. */
static XString GetTextFileContent( XString aFilePath )
{
  FILE*          file;
  long           size;
  XString        result;
  unsigned char* buf;

  /* Try to open the file. If the operation fails, return an empty string. 
     The file is searched relative to the current working directory. This
     corresponds to the directory of the opened Embedded Wizard project. */
  if ( _wfopen_s( &file, aFilePath, L"rt" ) != 0 )
    return 0;

  /* Query the size of the file in bytes (character). */
  fseek( file, 0, SEEK_END );
  size = ftell( file );
  fseek( file, 0, SEEK_SET );

  /* Reserve temp. memory for the file to load. */
  buf = EwAlloc( size );

  /* Load the file content into the buffer and close the file. */
  fread( buf, 1, size, file );
  fclose( file );

  /* Assuming, the loaded content is encoded in Utf8 format, create from it
     a new Embedded Wizard string. */
  result = EwNewStringUtf8( buf, size );

  /* Free the temp. used memory and ... */
  EwFree( buf );

  /* ... return the just created string. */
  return result;
}


/* Table describing all intrinsics implemented in this module */
EW_DEFINE_INTRINSICS
  EW_INTRINSIC
  (
    L"IntrinsicOpenFolder",
    L"void",
    2,
    L"string,string",
    L"aFolder,aPattern",
    OpenFolder
  )

  EW_INTRINSIC
  (
    L"IntrinsicCloseFolder",
    L"void",
    0,
    L"",
    L"",
    CloseFolder
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetNoOfFiles",
    L"int32",
    0,
    L"",
    L"",
    GetNoOfFiles
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetFileName",
    L"string",
    1,
    L"int32",
    L"aFileNo",
    GetFileName
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetTextFileContent",
    L"string",
    1,
    L"string",
    L"aFilePath",
    GetTextFileContent
  )
EW_END_OF_INTRINSICS


/* pba */