
#include "ewrte.h"

/* Include VC++ header files needed in this example. */
#include <stdio.h>
#include <wchar.h>
#include <io.h>
#include <windows.h>


EW_MODULE
( 
  INTRINSICS_IFC_VERSION,
  L"IntrinsicsModule", 
  L"Intrinsics module to access the file system"
)


/* Following type definition determines the data structure where collected file
   names are stored. */
typedef struct
{
  int       NoOfFiles;
  wchar_t** Files;
} FileNames;


/* The function frees the information collected by the preceding OpenFolder()
   invocation and passed in the parameter aHandle. */
static void CloseFolder( XHandle aHandle )
{
  FileNames* fileNames = (FileNames*)aHandle;
  int        fileNo    = 0;

  /* No collected files. */
  if ( !fileNames )
    return;

  /* Release the memory reserved for the collected file names. */
  for ( ; fileNo < fileNames->NoOfFiles; fileNo++ )
    if ( fileNames->Files[ fileNo ])
      EwFree( fileNames->Files[ fileNo ]);

  /* Release the memory reserved for the array with file names as well as
     for the data structure. */
  EwFree( fileNames->Files );
  EwFree( fileNames );
}


/* The function evaluates the given folder and collects all files matching 
   aPattern in the global array 'Files'. The folder is searched relative to
   the directory where the Embedded Wizard project is found. The function 
   returns a pointer to a data structure containing the collected names. */
static XHandle OpenFolder( XString aFolder, XString aPattern )
{
  wchar_t             path[ MAX_PATH ];
  intptr_t            handle;
  struct _wfinddata_t fileInfo;
  int                 noOfFiles = 0;
  int                 fileNo    = 0;
  FileNames*          fileNames = 0;

  /* The folder is searched relative to the current working directory. This
     corresponds to the directory of the opened Embedded Wizard project.
     Limit to files matching the pattern. */
  wcscpy_s( path, MAX_PATH, aFolder );
  wcscat_s( path, MAX_PATH, L"\\" );
  wcscat_s( path, MAX_PATH, aPattern );

  /* Phase 1: count the files matching the pattern. */
  if (( handle = _wfindfirst( path, &fileInfo )) != -1 )
  {
    do
    {
      /* Is this a file? */
      if ( !( fileInfo.attrib & _A_SUBDIR ))
        noOfFiles++;
    }
    while (( _wfindnext( handle, &fileInfo ) == 0 ));

    /* Terminate the search operation and release unused memory. */
    _findclose( handle );
  }

  /* Phase 2: Allocate memory for a data structure and an array where the
     file names will be stored. */
  fileNames            = EwAlloc( sizeof( FileNames ));
  fileNames->Files     = EwAlloc( noOfFiles * sizeof( XString ));
  fileNames->NoOfFiles = noOfFiles;
  memset( fileNames->Files, 0, noOfFiles * sizeof( XString ));

  /* Phase 3: Collect the file names matching the pattern. */
  if (( handle = _wfindfirst( path, &fileInfo )) != -1 )
  {
    do
    {
      /* Is this a file? Collect it in the array 'Files' */
      if ( !( fileInfo.attrib & _A_SUBDIR ) && ( fileNo < noOfFiles ))
      {
        int len = wcslen( fileInfo.name );

        fileNames->Files[ fileNo ] = EwAlloc(( len + 1 ) * sizeof( wchar_t ));
        wcscpy_s( fileNames->Files[ fileNo++ ], len + 1, fileInfo.name );
      }
    }
    while (( _wfindnext( handle, &fileInfo ) == 0 ));

    /* Terminate the search operation and release unused memory. */
    _findclose( handle );
  }

  /* Return a pointer to the data structure containing the collected files. */
  return (XHandle)fileNames;
}


/* The following function just returns how many files have been collected by the
   preceding OpenFolder() invocation and passed in data structure aHandle. */
static XInt32 GetNoOfFiles( XHandle aHandle )
{
  FileNames* fileNames = (FileNames*)aHandle;

  /* No collected files. */
  if ( !fileNames )
    return 0;

  return fileNames->NoOfFiles;
}


/* The following function returns the collected file with the given number. If
   the file is not available, the function returns an empty string. */
static XString GetFileName( XHandle aHandle, XInt32 aFileNo )
{
  FileNames* fileNames = (FileNames*)aHandle;

  /* No collected files. */
  if ( !fileNames )
    return 0;

  /* The requested file is not existing. Return an empty string. */
  if (( aFileNo < 0 ) || ( aFileNo >= fileNames->NoOfFiles ))
    return 0;

  /* Return an Embedded Wizard string containing a copy of the collected file
     name. */
  return EwNewString( fileNames->Files[ aFileNo ]);
}


/* The following function opens the given text file, reads its content and 
   returns it as a string. The parameter aFilePath specifies the location of
   the file relative to the directory where the Embedded Wizard project is 
   found. */
static XString GetTextFileContent( XString aFilePath )
{
  FILE*          file;
  long           size;
  XString        result;
  unsigned char* buf;

  /* Try to open the file. If the operation fails, return an empty string. 
     The file is searched relative to the current working directory. This
     corresponds to the directory of the opened Embedded Wizard project. */
  if ( _wfopen_s( &file, aFilePath, L"rt" ) != 0 )
    return 0;

  /* Query the size of the file in bytes (character). */
  fseek( file, 0, SEEK_END );
  size = ftell( file );
  fseek( file, 0, SEEK_SET );

  /* Reserve temp. memory for the file to load. */
  buf = EwAlloc( size );

  /* Load the file content into the buffer and close the file. */
  fread( buf, 1, size, file );
  fclose( file );

  /* Assuming, the loaded content is encoded in Utf8 format, create from it
     a new Embedded Wizard string. */
  result = EwNewStringUtf8( buf, size );

  /* Free the temp. used memory and ... */
  EwFree( buf );

  /* ... return the just created string. */
  return result;
}


/* Table describing all intrinsics implemented in this module */
EW_DEFINE_INTRINSICS
  EW_INTRINSIC
  (
    L"IntrinsicOpenFolder",
    L"handle",
    2,
    L"string,string",
    L"aFolder,aPattern",
    OpenFolder
  )

  EW_INTRINSIC
  (
    L"IntrinsicCloseFolder",
    L"void",
    1,
    L"handle",
    L"aHandle",
    CloseFolder
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetNoOfFiles",
    L"int32",
    1,
    L"handle",
    L"aHandle",
    GetNoOfFiles
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetFileName",
    L"string",
    2,
    L"handle,int32",
    L"aHandle,aFileNo",
    GetFileName
  )

  EW_INTRINSIC
  (
    L"IntrinsicGetTextFileContent",
    L"string",
    1,
    L"string",
    L"aFilePath",
    GetTextFileContent
  )
EW_END_OF_INTRINSICS


/* pba */