
#include "ewrte.h"
#include "ewgfx.h"

/* Include the QR-code generator library */
#include "qrcodegen.h"

/* Include additional C lib functions needed in the intrinsics */
#include <malloc.h>
#include <string.h>


EW_MODULE
( 
  INTRINSICS_IFC_VERSION,
  L"IntrinsicsModule", 
  L"Extern Bitmap intrinsics module"
)


/* The function creates a new ALPHA8 bitmap with a QR-code generated from the
   given text aName. */
XBitmap* EwLoadExternBitmap( XString aName )
{
  static uint8_t qr0[ qrcodegen_BUFFER_LEN_MAX ];
  static uint8_t tempBuffer[ qrcodegen_BUFFER_LEN_MAX ];
  static char    text[ 4296 ];
  int            len = EwGetStringLength( aName );
  XBitmap*       bitmap;
  XPoint         frameSize;
  XRect          lockArea;
  XBitmapLock*   lock;
  unsigned char* dest;
  int            ofs;
  int            x, y;
  int            scale = 1;

  /* No text to encode or the text is too long? */
  if ( !len || ( len > ( sizeof( text ) - 1 )))
    return 0;

  /* Convert the wide-char text in ASCII text. */
  EwStringToAnsi( aName, text, sizeof( text ), '?' );

  /* Generate the QR-code. */
  if ( !qrcodegen_encodeText( text, tempBuffer, qr0, qrcodegen_Ecc_MEDIUM, 
        qrcodegen_VERSION_MIN, qrcodegen_VERSION_MAX, qrcodegen_Mask_AUTO, 1 ))
    return 0;

  /* Determine a scaling factor for the resulting image to fill the entire 360
     pixel as good as possible. */
  scale = 360 / qrcodegen_getSize( qr0 );

  if ( scale < 1 )
    scale = 1;

  /* Get the size of the QR-image in pixel */
  frameSize.X       = qrcodegen_getSize( qr0 ) * scale;
  frameSize.Y       = qrcodegen_getSize( qr0 ) * scale;
  lockArea.Point1.X = 0;
  lockArea.Point1.Y = 0;
  lockArea.Point2.X = frameSize.X;
  lockArea.Point2.Y = frameSize.Y;

  /* Create an Embedded Wizard bitmap to store the QR image pixel data */
  bitmap = EwCreateBitmap( EW_PIXEL_FORMAT_ALPHA8, frameSize, 0, 1 );

  /* Could not create the image. */
  if ( !bitmap )
    return 0;

  /* Obtain access to the bitmap. */
  lock = EwLockBitmap( bitmap, 0, lockArea, 0, 1 );
  dest = (unsigned char*)lock->Pixel1;
  ofs  = lock->Pitch1Y - frameSize.X;

  /* Fill the bitmap with QR image content. */
  for ( y = 0; y < frameSize.Y; y++, dest += ofs )
    for ( x = 0; x < frameSize.X; x++, dest++ )
      *dest = ( qrcodegen_getModule( qr0, x / scale, y / scale ))?
              0xFF : 0x00;

  /* Release the lock */
  EwUnlockBitmap( lock );

  return bitmap;
}


/* Table describing all intrinsics implemented in this module */
EW_DEFINE_INTRINSICS
  EW_INTRINSIC
  (
    L"EwLoadExternBitmap",
    L"handle",
    1,
    L"string",
    L"aName",
    EwLoadExternBitmap
  )
EW_END_OF_INTRINSICS


/* Declaration of a data structure intended to store a copy of all globally
   existing data members permitting multiple instances of the module to coexist
   simultanously. Since this example uses Graphics Engine, it is important to
   store the pointers to the Graphics Engine module in the context. */
typedef struct
{
  XGraphicsEngineAPI GraphicsEngineAPI;
}
YourContextStruct;


/* At the initialization time of the module, create a data structure to store
   there a copy of global data members. */
EW_ENTRYPOINT void* EwInitModule( void )
{
  YourContextStruct* context;
  
  context = malloc( sizeof( YourContextStruct ));
  memset( context, 0, sizeof( YourContextStruct )); 
  return context; 
}


/* At the de-initialization time of the module, free the data structure used to
   store a copy of global data members. */
EW_ENTRYPOINT void EwDoneModule( void* aContext )
{
  free( aContext );
}


/* After all intrinsic modules are loaded, resolve all references to the Graphics
   Engine API functions */
EW_ENTRYPOINT void EwLinkModule( XModuleContext* aList )
{
  /* This intrinsic module requires access to the Graphics Engine API. Link this
     module with the GRaphics Engine API. */
  EwLinkWithGraphicsEngine( aList );
}


/* Just before Chora interpreter invokes an intrinsic, restore the state of all
   relevant (global) variables from the context data structure. */
EW_ENTRYPOINT void EwIntrinsicsProlog( void* aContext )
{
  YourContextStruct* context = (YourContextStruct*)aContext;

  /* Since this example uses Graphics Engine, it is important to store the
     pointers to the Graphics Engine module in the context. */
  EwGraphicsEngineAPI = context->GraphicsEngineAPI; 
}


/* Just after Chora interpreter invoked an intrinsic, store the state of all
   relevant (global) variables in the context data structure. */
EW_ENTRYPOINT void EwIntrinsicsEpilog( void* aContext )
{
  YourContextStruct* context = (YourContextStruct*)aContext;

  /* Since this example uses Graphics Engine, it is important to store the
     pointers to the Graphics Engine module in the context. */
  context->GraphicsEngineAPI = EwGraphicsEngineAPI;
}


/* pba */